import rpyc
from rpyc.utils.server import ThreadedServer
import logging

logging.basicConfig(level=logging.INFO, format="%(asctime)s [%(levelname)s] %(message)s")
logger = logging.getLogger(__name__)


class Calculator:
    """A simple calculator providing arithmetic operations"""

    def __init__(self):
        self.history = []

    def add(self, a, b):
        logger.info(f"add({a}, {b})")
        result = a + b
        self.history.append(f"{a} + {b} = {result}")
        return result

    def subtract(self, a, b):
        logger.info(f"subtract({a}, {b})")
        result = a - b
        self.history.append(f"{a} - {b} = {result}")
        return result

    def multiply(self, a, b):
        logger.info(f"multiply({a}, {b})")
        result = a * b
        self.history.append(f"{a} * {b} = {result}")
        return result

    def divide(self, a, b):
        logger.info(f"divide({a}, {b})")
        if b == 0:
            logger.error("Divide by zero")
            raise ValueError("Cannot divide by zero")
        result = a / b
        self.history.append(f"{a} / {b} = {result}")
        return result

    def get_history(self):
        logger.info("get_history()")
        return self.history


class CalculatorService(rpyc.Service):
    """RPyC service exposing calculator methods"""

    def on_connect(self, conn):
        logger.info(f"Client connected: {conn}")
        self.calculator = Calculator()

    def on_disconnect(self, conn):
        logger.info(f"Client disconnected: {conn}")

    # Expose operations directly on the service root
    def exposed_add(self, a, b):
        return self.calculator.add(a, b)

    def exposed_subtract(self, a, b):
        return self.calculator.subtract(a, b)

    def exposed_multiply(self, a, b):
        return self.calculator.multiply(a, b)

    def exposed_divide(self, a, b):
        return self.calculator.divide(a, b)

    def exposed_get_history(self):
        return self.calculator.get_history()


def main():
    logger.info("RPyC server listening on port 18861")
    server = ThreadedServer(CalculatorService, port=18861)

    try:
        server.start()
    except KeyboardInterrupt:
        logger.info("Shutting down server")


if __name__ == "__main__":
    main()
